-- **************************************************************************
--  This is the second mixer
--				2013,12,28 by Y nishimura
--					SSBmode 00:AM 01:USB 10:LSB 11:CW
-- **************************************************************************
LIBRARY ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
LIBRARY lpm;
use lpm.lpm_components.all;

entity Mixer is
	port
		(
			Hclk		: in  std_logic;	-- Master 4fsc clock
			Reset		: in  std_logic;	-- Master Reset
			Ptt			: in  std_logic;	-- PTT mode
			SSBmode		: in  std_logic_vector(1 downto 0);
			IFband		: in  std_logic_vector(1 downto 0);
			IFshift		: in  std_logic_vector(15 downto 0);
			Frequency	: in  std_logic_vector(25 downto 0); -- Tune frequency
			ADdata		: in  std_logic_vector(13 downto 0);
			Tepa		: out std_logic;
			Iout		: out std_logic_vector(15 downto 0);
			Qout		: out std_logic_vector(15 downto 0)
		);
end Mixer;

architecture norm of Mixer is
	signal	Ifull,Qfull			: std_logic_vector(35 downto 0);
	signal	Mulol,Muloh			: std_logic_vector(35 downto 0);
	signal	DDSda,Facc			: std_logic_vector(29 downto 0);
	signal	Freque				: std_logic_vector(25 downto 0);
	signal	Inad				: std_logic_vector(17 downto 0);
	signal	MulDaa,MulDab		: std_logic_vector(17 downto 0);
	signal	FullSin,FullCos		: std_logic_vector(17 downto 0);
	signal	NoiseI,NoiseQ,Freq	: std_logic_vector(15 downto 0);

	component SinCosS port
		(
			Hclk		: in  std_logic;	-- Master 4fsc clock
			Reset		: in  std_logic;	-- Master Reset
			Phase		: in  std_logic_vector(19 downto 0);
			Xaxis		: out std_logic_vector(17 downto 0);
			Yaxis		: out std_logic_vector(17 downto 0)
		);
	end component;

begin
	Tepa <= Facc(29);
-----------------------------------------------------------------------------------------
--		SSB frequency offset
-----------------------------------------------------------------------------------------
	process(Hclk,Reset)
		variable Tempb : std_logic_vector(2 downto 0);
		variable Tempa : std_logic_vector(15 downto 0);
		
	begin
		if Reset = '0' 		then Freq <= "0000000000000000";
		elsif rising_edge(Hclk) then
			Tempb(2) := SSBmode(1);
			Tempb(1 downto 0) := IFband;
			if (SSBmode = "00") or (SSBmode = "11")	then
--				Tempa := "0000000000000000";
				Tempa := "0000000000010100";	-- CW mode +20Hz ikeda
			else
				case Tempb is
					when "011" => Tempa := "1100010100000100";  -- -15.1KHz
					when "010" => Tempa := "1110001001010000";  -- -7.6KHz
					when "001" => Tempa := "1111001111100100";  -- -3.1KHz
--					when "000" => Tempa := "1111101010001000";  -- -1.4KHz
					when "000" => Tempa := "1111101010011100";  -- -1.38KHz ikeda
					when "111" => Tempa := "0011101011111100";  -- +15.1KHz
					when "110" => Tempa := "0001110110110000";  -- +7.6KHz
					when "101" => Tempa := "0000110000011100";  -- +3.1KHz
--					when "100" => Tempa := "0000010101111000";  -- +1.4KHz
					when "100" => Tempa := "0000010110001100";  -- +1.42KHz ikeda
					when others => Tempa := "0000000000000000";
				end case;
			end if;
			Freq <= IFshift + Tempa;
		end if;
	end process;

-----------------------------------------------------------------------------------------
--		IFshift
-----------------------------------------------------------------------------------------
	process(Hclk,Reset)
		variable Tempa : std_logic_vector(25 downto 0);

	begin
		if Reset = '0' 		then Freque <= "00000000000000000000000000";
		elsif rising_edge(Hclk) then
			if Freq(15) = '1'	then Tempa(25 downto 16) := "1111111111";
								else Tempa(25 downto 16) := "0000000000";
			end if;
			Tempa(15 downto 0) := Freq;
			Freque <= Frequency + Tempa;
		end if;
	end process;

-----------------------------------------------------------------------------------------
--		1Hz --> DDS data   1Hz = 2^16/65MHz 
--						offset 2^27
-----------------------------------------------------------------------------------------
	MulDaa(17 downto 15) <= "000";
	MulDaa(14 downto 0) <= Freque(14 downto 0);
	U15: lpm_mult GENERIC map (
			LPM_WIDTHA => 18,LPM_WIDTHB => 18,LPM_WIDTHP => 36,
			LPM_REPRESENTATION => "UNSIGNED",LPM_PIPELINE => 2) 
  	    PORT map (
					dataa => MulDaa,
					datab => "100001000010011100",   -- 2^43/65MHz
					aclr  => '0',
					clock => Hclk,
					clken => '1',
        			result => Mulol
				 ); 

	MulDab(17 downto 11) <= "0000000";
	MulDab(10 downto 0) <= Freque(25 downto 15);
	U16: lpm_mult GENERIC map (
			LPM_WIDTHA => 18,LPM_WIDTHB => 18,LPM_WIDTHP => 36,
			LPM_REPRESENTATION => "UNSIGNED",LPM_PIPELINE => 2) 
  	    PORT map (
					dataa => MulDab,
					datab => "100001000010011100",
					aclr  => '0',
					clock => Hclk,
					clken => '1',
        			result => Muloh
				 ); 

	process(Hclk,Reset)
		variable Tempa,Tempb : std_logic_vector(29 downto 0);
		
	begin
		if Reset = '0' then DDSda <= "000000000000000000000000000000";
		elsif rising_edge(Hclk) then
			Tempa(22 downto 0) := Mulol(35 downto 13);
			Tempa(29 downto 23) := "0000000";
			Tempb(29 downto 2) := Muloh(27 downto 0); Tempb(1 downto 0) := "00";
			DDSda <= Tempa + Tempb;
		end if;
	end process;

------------------------------------------------------------------------------
--		Address Gen
------------------------------------------------------------------------------
	process(Hclk,Reset)
	begin
		if Reset = '0' 		then Facc <= "000000000000000000000000000000";
		elsif rising_edge(Hclk) then
			Facc <= Facc + DDSda;
		end if;
	end process;

------------------------------------------------------------------------------
--		Sin / Cos gen by CORDIC
------------------------------------------------------------------------------
	U1: SinCosS port map
		(
			Hclk	=> Hclk,	-- Master 4fsc clock
			Reset	=> Reset,	-- Master Reset
			Phase	=> Facc(29 downto 10),
			Xaxis	=> FullSin,	
			Yaxis	=> FullCos
		);

----------------------------------------------------------------------------------
--		ADC de-scrambler
----------------------------------------------------------------------------------
	process(Hclk,Reset)
		variable Tempa : std_logic_vector(13 downto 0);
		
	begin
		if Reset = '0' then Inad <= "000000000000000000";
		elsif rising_edge(Hclk) then
			if ADdata(0) = '0'	then Tempa := "00000000000000";
								else Tempa := "11111111111110";
			end if;
			if Ptt = '1'		then Inad(17 downto 4) <= "00000000000000";
								else Inad(17 downto 4) <= ADdata xor Tempa;
			end if;
			Inad(3 downto 0) <= "0000";
		end if;
	end process;

--	process(ADdata)  -- De-scrambler
--		variable Tempa : std_logic_vector(13 downto 0);
	
--	begin
--		if ADdata(0) = '0'		then Tempa := "00000000000000";
--								else Tempa := "11111111111110";
--		end if;
--		Inad(17 downto 4) <= ADdata xor Tempa;
--		Inad(3 downto 0) <= "0000";
--	end process;

----------------------------------------------------------------------------------
--		I/Q mixer
----------------------------------------------------------------------------------
	U10: lpm_mult GENERIC map (
			LPM_WIDTHA => 18,LPM_WIDTHB => 18,LPM_WIDTHP => 36,
			LPM_REPRESENTATION => "SIGNED",LPM_PIPELINE => 2) 
  	    PORT map (
					dataa => Inad,
					datab => FullCos,
					aclr  => '0',
					clock => Hclk,
					clken => '1',
        			result => Ifull
				 ); 

	U11: lpm_mult GENERIC map (
			LPM_WIDTHA => 18,LPM_WIDTHB => 18,LPM_WIDTHP => 36,
			LPM_REPRESENTATION => "SIGNED",LPM_PIPELINE => 2) 
  	    PORT map (
					dataa => Inad,
					datab => FullSin,
					aclr  => '0',
					clock => Hclk,
					clken => '1',
        			result => Qfull
				 ); 

-------------------------------------------------------------------------------------------
--		Noise Shaping
-------------------------------------------------------------------------------------------
 	process(Hclk,Reset)
		variable Tempa,Tempb,Tempc,Tempd : std_logic_vector(15 downto 0);
		
	begin
		if Reset = '0' then 
			NoiseI <= "0000000000000000"; NoiseQ <= "0000000000000000";
		elsif rising_edge(Hclk) then
			Tempa(15) := '0'; Tempa(14 downto 0) := NoiseI(14 downto 0);
			Tempb(15) := '0'; Tempb(14 downto 0) := NoiseQ(14 downto 0);
			Tempc(15) := '0'; Tempc(14 downto 0) := Ifull(18 downto 4);
			Tempd(15) := '0'; Tempd(14 downto 0) := Qfull(18 downto 4);
			NoiseI <= Tempa + Tempc;
			NoiseQ <= Tempb + Tempd;
		end if;
	end process;

-------------------------------------------------------------------------------------------
--		Output process
-------------------------------------------------------------------------------------------
 	process(Hclk,Reset)
	begin
		if Reset = '0' then 
			Iout <= "0000000000000000";	Qout <= "0000000000000000";
		elsif rising_edge(Hclk) then
			if NoiseI(15) = '1'	then Iout <= Ifull(34 downto 19) + 1;
								else Iout <= Ifull(34 downto 19);
			end if;
			if NoiseQ(15) = '1'	then Qout <= Qfull(34 downto 19) + 1;
								else Qout <= Qfull(34 downto 19);
			end if;
		end if;
	end process;

end norm;
