-- **************************************************************************
--  This is the Modulation and Oversampling to the AD9957
--				(C)Copyright 2014,03,17 by Y nishimura
--				1GHz/(62*4)/128=31.502016KHz sample
-- **************************************************************************
LIBRARY ieee;
use ieee.std_logic_1164.all;
use ieee.std_logic_unsigned.all;
LIBRARY lpm;
use lpm.lpm_components.all;

entity TransmitTotal is
	port
		(
			Hclk		: in  std_logic;	-- 65MHz
			Reset		: in  std_logic;	-- Master Reset
			Ptt			: in  std_logic;	-- Ptt
			TRXmode		: in  std_logic;	-- Test TRX mode
			AdcSamp		: in  std_logic;	-- ADC sampling clock
			RstCIC		: in  std_logic;	-- Reset CIC fitler reg.
			PDclk		: in  std_logic;	-- I/Q clock from AD9957
			Test		: in  std_logic;	-- 1kHz Test signal input
			CWkey		: in  std_logic;	-- CW key input
			ALCcon		: in  std_logic_vector(2 downto 0);	-- ALC control
			Ifband		: in  std_logic_vector(1 downto 0);	-- Filter band choice
			SSBmode		: in  std_logic_vector(1 downto 0);	-- 00:AM 01:LSB  10:USB 11:CW
			Indata		: in  std_logic_vector(15 downto 0); -- MIC input
			Amod		: in  std_logic_vector(7 downto 0);  -- AM modulation
			ALCcont		: out std_logic_vector(6 downto 0);
			Dgate		: out std_logic;	-- Output Sampling gate
			Pulse		: out std_logic;
			CWmod		: out std_logic;
			Gate31		: out std_logic;	-- 31kHz sample gate
			IQout		: out std_logic_vector(17 downto 0)	-- I/Q out
		);
end TransmitTotal;

architecture norm of TransmitTotal is
	signal	FoutQ,AlcoutI,AlcoutQ	: std_logic_vector(35 downto 0);
	signal	Iina,Qina,Sin			: std_logic_vector(30 downto 0);
	signal	Ioa,Qoa,Outf			: std_logic_vector(30 downto 0);
	signal	HilI,HilQ				: std_logic_vector(17 downto 0);
	signal	Depth,FmI,FmQ,LoutQ		: std_logic_vector(17 downto 0);
	signal	DaOutI,DaOutQ,AccQ		: std_logic_vector(17 downto 0);
	signal	IDout,QDout,ICout,QCout	: std_logic_vector(17 downto 0);
	signal	AlcinI,AlcinQ,Alc		: std_logic_vector(17 downto 0);
	signal	Tests,Testsg			: std_logic_vector(15 downto 0);
	signal	Emic,Sinda,Cdata		: std_logic_vector(15 downto 0);
	signal	TxScnt					: std_logic_vector(10 downto 0);
	signal	Csta					: std_logic_vector(4 downto 0);
	signal	Bsta					: std_logic_vector(1 downto 0);
	signal	AM,CWmode,USBsel		: std_logic;
	signal	Pgate,Fmmode,FourGt		: std_logic;
	signal	Bsamp,Hsamp,Spul,Swiq	: std_logic;

	component FirFront is
		port
			(
				Hclk		: in  std_logic;	-- 65MHz
				Reset		: in  std_logic;	-- Master Reset
				CWkey		: in  std_logic;	-- CW key input
				CWmode		: in  std_logic;	-- CW mode
				Cgate		: in  std_logic;	-- 1/1024
				Ifband		: in  std_logic; 	-- 1:15K  0:6K
				IdatIn		: in  std_logic_vector(30 downto 0);
				Dgate		: out std_logic;
				DaOutI		: out std_logic_vector(30 downto 0)
			);
	end component;

	component hilbert is
		port
			(
				Hclk		: in  std_logic;	-- 65MHz
				Reset		: in  std_logic;	-- Master Reset
				Cgate		: in  std_logic;	-- 1/1024
				SelUSB		: in  std_logic;	-- USB select
				IdatIn		: in  std_logic_vector(30 downto 0);
				Dgate		: out std_logic;
				DaOutI		: out std_logic_vector(17 downto 0);
				DaOutQ		: out std_logic_vector(17 downto 0)
			);
	end component;

	component Fmmod is
		port
			(
				Hclk		: in  std_logic;	-- 65MHz
				Reset		: in  std_logic;	-- Master Reset
				Sgate		: in  std_logic;	-- Sampling timing
				Mdata		: in  std_logic_vector(19 downto 0);
				Xaxis		: out std_logic_vector(17 downto 0);
				Yaxis		: out std_logic_vector(17 downto 0)
			);
	end component;

	component Emphasys is
		port
			(
				Mclk		: in  std_logic;	-- 65MHz
				Reset		: in  std_logic;	-- Master Reset
				Cgate		: in  std_logic;	-- Sample gate
				FMmode		: in  std_logic;	-- 1:FM mode
				Mic			: in  std_logic_vector(15 downto 0); -- SH bus input
				Dout		: out std_logic_vector(15 downto 0)
			);
	end component;

	component Over4times is
		port
			(
				Hclk		: in  std_logic;	-- 65MHz clock
				Reset		: in  std_logic;	-- Master Reset
				Test		: in  std_logic;	-- Test signal for sim
				PDclk		: in  std_logic;	-- 1GHz/248=4.03225MHz(I/Q sampling)
				IdatIn		: in  std_logic_vector(17 downto 0);
				QdatIn		: in  std_logic_Vector(17 downto 0);
				Dgate		: out std_logic;	-- 4 times sampling gate
				Sample		: out std_logic;	-- Base sampling gate
				Hsamp		: out std_logic;	-- 4.03225MHz sampling gate
				Pulse		: out std_logic;
				Bstate		: out std_logic_vector(1 downto 0);  -- Base clock timing
				DaOutI		: out std_logic_vector(17 downto 0); -- Comb filter out
				DaOutQ		: out std_logic_vector(17 downto 0)  -- Comb filter out
			);
	end component;

	component OverCICi is
		port
			(
				Hclk		: in  std_logic;	-- 65MHz
				Reset		: in  std_logic;	-- Master Reset
				CCIrst		: in  std_logic;
				IBuf		: in  std_logic_vector(17 downto 0); -- I/Q data
				Rtim		: in  std_logic;	-- Base timing = 1GHz/248/128*4 (4 times)
				Hsamp		: in  std_logic;	-- High rate = 1GHz/248
				DaOut		: out std_logic_vector(17 downto 0)  -- Comb filter out
			);
	end component;

	component OverCICq is
		port
			(
				Hclk		: in  std_logic;	-- 65MHz
				Reset		: in  std_logic;	-- Master Reset
				CCIrst		: in  std_logic;
				IBuf		: in  std_logic_vector(17 downto 0); -- I/Q data
				Rtim		: in  std_logic;	-- Base timing = 1GHz/248/128*4 (4 times)
				Hsamp		: in  std_logic;	-- High rate = 1GHz/248
				DaOut		: out std_logic_vector(17 downto 0)  -- Comb filter out
			);
	end component;

	component FrontOver is
		port
			(
				Hclk		: in  std_logic;	-- 65MHz
				Reset		: in  std_logic;	-- Master Reset
				Txgate		: in  std_logic;	-- Tx sampling gate 31.5kHz
				Adgate		: in  std_logic;	-- ADC gate 31.7kHz
				Indata		: in  std_logic_vector(15 downto 0); -- ADC data
				Outdata		: out std_logic_vector(15 downto 0)
			);
	end component;

begin
	Gate31 <= Bsamp;
------------------------------------------------------------------------------
--		Front end filtering
------------------------------------------------------------------------------
	CWmod <= CWmode;
	process(Hclk,Reset)
	begin
		if Reset = '0' 		then CWmode <= '0'; USBsel <= '0'; Fmmode <= '0';
		elsif rising_edge(Hclk) then
			if (SSBmode = "11") and (Ifband = "11")		then CWmode <= '1';
														else CWmode <= '0';
			end if;
			if (SSBmode = "11") and (Ifband = "10")		then Fmmode <= '1';
														else Fmmode <= '0';
			end if;
			if SSBmode = "10" 		then USBsel <= '1';
									else USBsel <= '0';
			end if;
		end if;
	end process;

------------------------------------------------------------------------------
--		Test signal generator
------------------------------------------------------------------------------
	process(Hclk,Reset)
	begin
		if Reset = '0' 		then Csta <= "00000";
		elsif rising_edge(Hclk) then
			if AdcSamp = '1'	then
				if Csta(3 downto 0) = "1001"	then
					 Csta(3 downto 0) <= "0000"; Csta(4) <= not Csta(4);
				else Csta <= Csta + 1;
				end if;
			else Csta <= Csta;
			end if;
		end if;
	end process;
	
	process(Hclk,Reset)
	begin
		if Reset = '0' 		then
			Testsg <= "0000000000000000"; Tests <= "0000000000000000";
		elsif rising_edge(Hclk) then
			if AdcSamp = '1'	then
				case Csta(3 downto 0) is
					when "0000" => Testsg <= "0010010100010101";
					when "0001" => Testsg <= "0100011010001000";
					when "0010" => Testsg <= "0110000100010101";
					when "0011" => Testsg <= "0111001000100000";
					when "0100" => Testsg <= "0111100000000000";
					when "0101" => Testsg <= "0111001000100000";
					when "0110" => Testsg <= "0110000100010101";
					when "0111" => Testsg <= "0100011010001000";
					when "1000" => Testsg <= "0010010100010101";
					when "1001" => Testsg <= "0000000000000000";
					when others => Testsg <= "0000000000000000";
				end case;
				if Csta(4) = '0' 	then Tests <= Testsg;
									else Tests <= (not Testsg) + 1;
				end if;
			else Testsg <= Testsg; Tests <= Tests;
			end if;
		end if;
	end process;
	
	process(Hclk,Reset)
	begin
		if Reset = '0' 		then
			Sinda <= "0000000000000000"; TxScnt <= "00000000000";
		elsif rising_edge(Hclk) then
			if AdcSamp = '1'	then
				if (Ptt = '0') and (TRXmode = '0')	then
					Sinda <= "0000000000000000";
					TxScnt <= "00000000000";
				else
					if Test = '1'	then Sinda <= Tests;
					else
						if TxScnt(10 downto 9) = "11"	then
							 Sinda <= Indata;
						else Sinda <= "0000000000000000";
						end if;
					end if;
					if TxScnt(10 downto 9) = "11"	then
						 TxScnt <= TxScnt;
					else TxScnt <= TxScnt + 1;
					end if;
				end if;
			else TxScnt <= TxScnt; Sinda <= Sinda;
			end if;
		end if;
	end process;

	U0: FrontOver port map
			(
				Hclk		=> Hclk,	-- 65MHz
				Reset		=> Reset,	-- Master Reset
				Txgate		=> Bsamp,	-- Tx sampling gate 31.5kHz
				Adgate		=> AdcSamp,	-- ADC gate 31.7kHz
				Indata		=> Sinda, 	-- ADC data
				Outdata		=> Cdata
			);

	U1: Emphasys port map -- Pre emphasys for the NFM
			(
				Mclk		=> Hclk,		-- 65MHz
				Reset		=> Reset,		-- Master Reset
				Cgate		=> Bsamp,		-- Sample gate
				FMmode		=> Fmmode,		-- 1:FM mode
				Mic			=> Cdata,		
				Dout		=> Emic
			);

	Sin(30 downto 15) <= Emic; Sin(14 downto 0) <= "000000000000000";
	U2: FirFront port map
			(
				Hclk		=> Hclk,		-- 65MHz
				Reset		=> Reset,		-- Master Reset
				CWkey		=> CWkey,		-- CW key input
				CWmode		=> CWmode,		-- CW mode
				Cgate		=> Bsamp,		-- 31kHz sample gate
				Ifband		=> Ifband(0),	-- 1:7.5K  0:3K
				IdatIn		=> Sin,			-- MIC input
				Dgate		=> Dgate,		-- Output gate
				DaOutI		=> Outf
			);

------------------------------------------------------------------------------
--		Hilbert Filer
------------------------------------------------------------------------------
	U3: hilbert port map
			(
				Hclk		=> Hclk,		-- 65MHz
				Reset		=> Reset,		-- Master Reset
				Cgate		=> Bsamp,		-- 1/1024
				SelUSB		=> USBsel,		-- USB select
				IdatIn		=> Outf,		
				Dgate		=> Pgate,	
				DaOutI		=> HilI,
				DaOutQ		=> HilQ
			);

	
------------------------------------------------------------------------------
--		FM modulation
------------------------------------------------------------------------------
	U4: Fmmod port map
			(
				Hclk		=> Hclk,		-- 65MHz
				Reset		=> Reset,		-- Master Reset
				Sgate		=> Bsamp,		-- Sampling timing
				Mdata		=> FoutQ(34 downto 15),
				Xaxis		=> FmI,
				Yaxis		=> FmQ
			);

----------------------------------------------------------------------------
--		AM CW moduration depth
------------------------------------------------------------------------------
	process(Hclk,Reset)
	begin
		if Reset = '0' 		then Depth <= "000000000000000000";
		elsif rising_edge(Hclk) then
			if CWmode = '1'	then
				Depth(17) <= '0';
				Depth(16 downto 9) <= Amod;
				depth(8 downto 0) <= "111111111";
			else
				if Fmmode = '1'	then Depth <= "000111000111000101"; -- FM dev 2^18*3.5k/31.5kHz
				else
					Depth(17 downto 16) <= "00";
					Depth(15 downto 8) <= Amod;
					Depth(7 downto 0) <= "11111111";
				end if;
			end if;
		end if;
	end process;
	
	U5: lpm_mult GENERIC map (
			LPM_WIDTHA => 18,LPM_WIDTHB => 18,LPM_WIDTHP => 36,
			LPM_REPRESENTATION => "SIGNED",LPM_PIPELINE => 2) 
  	    PORT map (
					dataa => HilI,
					datab => Depth,
					aclr  => '0',
					clock => Hclk,
					clken => '1',
        			result => FoutQ
				 ); 
 
---------------------------------------------------------------------------------
--		Noise Shaping
---------------------------------------------------------------------------------
	process(Hclk,Reset)
		variable Tempa,Tempb : std_logic_vector(17 downto 0);
		
	begin
		if Reset = '0' 		then AccQ <= "000000000000000000";
		elsif rising_edge(Hclk) then
			Tempa(17) := '0';
			Tempa(16 downto 0) := FoutQ(16 downto 0);
			Tempb(17) := '0';
			Tempb(16 downto 0) := AccQ(16 downto 0);
			AccQ <= Tempa + Tempb;
		end if;
	end process;

	process(Hclk,Reset)
	begin
		if Reset = '0' 		then LoutQ <= "000000000000000000";
		elsif rising_edge(Hclk) then
			if AccQ(17) = '1'	then LoutQ <= FoutQ(34 downto 17) + 1;
								else LoutQ <= FoutQ(34 downto 17);
			end if;
		end if;
	end process;

-------------------------------------------------------------------------------
--		Modulation
-------------------------------------------------------------------------------
	process(Hclk,Reset)
	begin
		if Reset = '0' 		then
			DaOutI <= "000000000000000000"; DaOutQ <= "000000000000000000";
		elsif rising_edge(Hclk) then
			if (Ptt = '0') and (TRXmode = '0')	then
				DaOutI <= "000000000000000000";
				DaOutQ <= "000000000000000000";
			else
				if Fmmode = '1'	then
					DaOutI <= FmI; DaOutQ <= FmQ;
				else
					case SSBmode is
						when "00" => DaOutI <= LoutQ + "010000000000000000";
									 DaOutQ <= "000000000000000000";
						when "01" => DaOutI <= HilI; DaOutQ <= HilQ;
						when "10" => DaOutI <= HilI; DaOutQ <= HilQ;
						when "11" => DaOutI <= LoutQ;
									 DaOutQ <= "000000000000000000";
						when others => DaOutI <= "010000000000000000";
									   DaOutQ <= "010000000000000000";
					end case;
				end if;
			end if;
		end if;
	end process;

-------------------------------------------------------------------------------
--		ALC logic
-------------------------------------------------------------------------------	
	process(Hclk,Reset)
		variable Tempa,Tempb : std_logic_vector(17 downto 0);
		
	begin
		if Reset = '0' 		then
			AlcinI <= "000000000000000000";
			AlcinQ <= "000000000000000000";
			Alc <= "000000000000000000";
		elsif rising_edge(Hclk) then
			AlcinI <= DaOutI;  AlcinQ <= DaOutQ;
			Tempa(4 downto 0) := "00000";
			if ALCcon = "000" then
				Tempa(17 downto 5) := "0000000000001";
			else
				if ALCcon = "111"	then
					Tempa(17 downto 5) := "0000000000000";
				else
					Tempa(17 downto 9) := "111111111";
					Tempa(8 downto 6) := ALCcon;
				end if;
			end if;
			Tempb := Alc + Tempa;
			if (RstCIC = '0') or (TRXmode = '1') then
				Alc <= "011111111111111111";
			else
				if Bsamp = '1'	then
					if ALCcon = "001"		then Alc <= "000010000000000000";
					else
						if Tempb(17) = '1'	then Alc <= "011111111111111111";
						else
							if Tempb(16 downto 13) = "0000"	then
								 Alc <= "000010000000000000";
							else Alc <= Tempb;
							end if;
						end if;
					end if;
				else Alc <= Alc;
				end if;
			end if;
		end if;
	end process;
	ALCcont <= Alc(14 downto 8);

	U6: lpm_mult GENERIC map (
			LPM_WIDTHA => 18,LPM_WIDTHB => 18,LPM_WIDTHP => 36,
			LPM_REPRESENTATION => "SIGNED",LPM_PIPELINE => 2) 
  	    PORT map (
					dataa => AlcinI,
					datab => Alc,
					aclr  => '0',
					clock => Hclk,
					clken => '1',
        			result => AlcoutI
				 ); 
 
	U7: lpm_mult GENERIC map (
			LPM_WIDTHA => 18,LPM_WIDTHB => 18,LPM_WIDTHP => 36,
			LPM_REPRESENTATION => "SIGNED",LPM_PIPELINE => 2) 
  	    PORT map (
					dataa => AlcinQ,
					datab => Alc,
					aclr  => '0',
					clock => Hclk,
					clken => '1',
        			result => AlcoutQ
				 ); 
 

-------------------------------------------------------------------------------
--		35kHz x Four times over sampling 
-------------------------------------------------------------------------------
	U15: Over4times port map
			(
				Hclk		=> Hclk,		-- 65MHz clock
				Reset		=> Reset,		-- Master Reset
				Test		=> '0',			-- Test signal for sim
				PDclk		=> PDclk,		-- 1GHz/248=4.03225MHz(I/Q sampling)
				IdatIn		=> AlcoutI(34 downto 17),	-- 35kHz sampling I data input
				QdatIn		=> AlcoutQ(34 downto 17),	-- 35kHz sampling Q data input
				Dgate		=> FourGt,		-- 4 times sampling gate
				Sample		=> Bsamp,		-- Base sampling gate
				Hsamp		=> Hsamp,		-- 4.03225MHz sampling gate
				Pulse		=> Spul,
				Bstate		=> Bsta,		-- Base clock timing
				DaOutI		=> IDout,		-- Comb filter out
				DaOutQ		=> QDout		-- Comb filter out
			);
	Pulse <= Spul;
	
-----------------------------------------------------------------------------
--		CIC filter
--------------------------------------------------------------------------------
	U16: OverCICi port map
			(
				Hclk		=> Hclk,		-- Master 4fsc clock
				Reset		=> Reset,		-- Master Reset
				CCIrst		=> RstCIC,
				IBuf		=> IDout,		-- I/Q data
				Rtim		=> FourGt,		-- Base timing = 1GHz/248/128*4 (4 times)
				Hsamp		=> Hsamp,		-- High rate = 1GHz/248
				DaOut		=> ICout		-- Comb filter out
			);

	U17: OverCICq port map
			(
				Hclk		=> Hclk,		-- Master 4fsc clock
				Reset		=> Reset,		-- Master Reset
				CCIrst		=> RstCIC,
				IBuf		=> QDout,		-- I/Q data
				Rtim		=> FourGt,		-- Base timing = 1GHz/248/128*4 (4 times)
				Hsamp		=> Hsamp,		-- High rate = 1GHz/248
				DaOut		=> QCout		-- Comb filter out
			);

--------------------------------------------------------------------------------
--		Combiner
--------------------------------------------------------------------------------
	process(Hclk,Reset)
	begin
		if Reset = '0' 		then IQout <= "000000000000000000";
		elsif rising_edge(Hclk) then
			if PDclk = '1'	then IQout <= QCout;
							else IQout <= ICout;
			end if;
		end if;
	end process;

end norm;
